#functions to visualize and quantify polysome profiles from Biocomp fraction station
# for input (filename) uses csv file from Biocomp TRIAX software running in absorbance and fluoresence mode.
#written by Jarod Rollins at the Mount Desert Island Biological Laboratory * jrollins@mdibl.org *

#The function polyproAUC() determine the area under the curve for the different ribosomal forms as defined by the user.
#The area will be reported as a percent of the total area uner the curve
#notes on use - do NOT change graph size from the default (7"x7") before calculating the area under the curve 

#after function is run and plot is visualized, the user chooses 6 points on the graph with the mouse
	#1st click should be start of 40S
	#2nd click is between 40S and 60S
	#3rd click between 60S and 80S
	#4th click is start of polysome
	#5th click is middle of polysome
	#6th click is end of polysome

#packages needed 
install.packages('grid')
install.packages('MESS')


############################
### Start of function ######
############################
polyproAUC=function(filename,Output=''){
library(grid)
library(MESS)
#import data from gradient station
raw=read.table(file=filename, header=T,sep=',',skip=51)
#generate an empty plot with consistant dimentions
x11(width=7.78, height=7)
par(mar=c(5.1, 4.1, 4.1, 6))
#generate an plot based on abs at 254 nm
plot(x=raw$Position,y=raw$AbsA,type='n',ylab='Absorbance 254 nm (AU)',xlab='Distance (mm)',xlim=c(0,87.952800)) 
lines(raw$Position,raw$AbsA,col='black')
par(new=TRUE)
#overlay plot from flouresence
plot(x=raw$Position,y=raw$SampleFluor,col='red',ylab='',xlab='',axes=F,type='l',xlim=c(0,87.952800))
axis(4, ylim=c(0,max(raw$SampleFluor)),las=1)
mtext("Relative Fluoresence Units",side=4,line=3,col='red')


#mouse click start of 40S and convert graph pixels to distance in mm 
start.total=((as.numeric(grid.locator(unit = "cm"))-2.645)*6.5)[1]
#click end of 40S
end40=((as.numeric(grid.locator(unit = "cm"))-2.645)*6.5)[1]
#mouse click start of 60S 
end60=((as.numeric(grid.locator(unit = "cm"))-2.645)*6.5)[1]
#mouse click end of polysome (or other area of interest)
start.poly=((as.numeric(grid.locator(unit = "cm"))-2.645)*6.5)[1]
#mouse click end of polysome (or other area of interest)
mid.poly=((as.numeric(grid.locator(unit = "cm"))-2.645)*6.5)[1]
#mouse click end of polysome (or other area of interest)
stop.poly=((as.numeric(grid.locator(unit = "cm"))-2.645)*6.5)[1]

#calulate area under the curve for absorbance based on selected positons
RNA.AUC.40=auc(raw$Position,raw$AbsA,from=start.total,to=end40)
RNA.AUC.60=auc(raw$Position,raw$AbsA,from=end40,to=end60)
RNA.AUC.80=auc(raw$Position,raw$AbsA,from=end60,to=start.poly)
RNA.AUC.Low=auc(raw$Position,raw$AbsA,from=start.poly,to=mid.poly)
RNA.AUC.High=auc(raw$Position,raw$AbsA,from=mid.poly,to=stop.poly)
RNA.AUC.Poly=auc(raw$Position,raw$AbsA,from=start.poly,to=stop.poly)
RNA.AUC.All=auc(raw$Position,raw$AbsA,from=start.total,to=stop.poly)
#place values into a vector
RNA.AUC=c(RNA.AUC.40,RNA.AUC.60,RNA.AUC.80,RNA.AUC.Low,RNA.AUC.High,RNA.AUC.Poly)


#place values into a dataframe
Ratio=data.frame(c(RNA.AUC.40/RNA.AUC.All*100),c(RNA.AUC.60/RNA.AUC.All*100),c(RNA.AUC.80/RNA.AUC.All*100),c(RNA.AUC.Low/RNA.AUC.All*100),c(RNA.AUC.High/RNA.AUC.All*100),c(RNA.AUC.Poly/RNA.AUC.All*100),row.names=c('RNA'))
colnames(Ratio)=c('40S','60S','80S','Low','High','Polysome')

#calulate area under the curve for fluoresence based on selected positons
Fluor.AUC.40=auc(raw$Position,raw$SampleFluor,from=start.total,to=end40)
Fluor.AUC.60=auc(raw$Position,raw$SampleFluor,from=end40,to=end60)
Fluor.AUC.80=auc(raw$Position,raw$SampleFluor,from=end60,to=start.poly)
Fluor.AUC.Low=auc(raw$Position,raw$SampleFluor,from=start.poly,to=mid.poly)
Fluor.AUC.High=auc(raw$Position,raw$SampleFluor,from=mid.poly,to=stop.poly)
Fluor.AUC.Poly=auc(raw$Position,raw$SampleFluor,from=start.poly,to=stop.poly)
Fluor.AUC.All=auc(raw$Position,raw$SampleFluor,from=start.total,to=stop.poly)

#place values into a vector
Fluor.AUC=c(Fluor.AUC.40,Fluor.AUC.60,Fluor.AUC.80,Fluor.AUC.Low,Fluor.AUC.High,Fluor.AUC.Poly)

#place values into a dataframe
Ratio=data.frame(c('Position','RNA','Fluoresence'),c(start.total,RNA.AUC.40/RNA.AUC.All*100,Fluor.AUC.40/Fluor.AUC.All*100),c(end40,RNA.AUC.60/RNA.AUC.All*100,Fluor.AUC.60/Fluor.AUC.All*100),c(end60,RNA.AUC.80/RNA.AUC.All*100,Fluor.AUC.80/Fluor.AUC.All*100),
c(start.poly,RNA.AUC.Low/RNA.AUC.All*100,Fluor.AUC.Low/Fluor.AUC.All*100),c(mid.poly,RNA.AUC.High/RNA.AUC.All*100,Fluor.AUC.High/Fluor.AUC.All*100),c(stop.poly,RNA.AUC.Poly/RNA.AUC.All*100,Fluor.AUC.Poly/Fluor.AUC.All*100))
colnames(Ratio)=c('Value','40S','60S','80S','Low','High','Polysome')

if( Output=='Excel'){
write.excel(Ratio)} else
#output AUC for each fraction to console
return(Ratio)

}

######End of function ######


###########Start of function to write output to clipboard
##Function from https://nhsrcommunity.com/blog/r-tip-copy-and-paste-data-from-to-excel/
write.excel <- function (x,row.names=F,col.names=TRUE,...) {
write.table (x,"clipboard",sep="\t",row.names=row.names,col.names=col.names,...)
}

